! Program to calculate the spectral albedo from the two spectrometer instrument.
! Written by Pat Arnott, April 2013.
! The 2 spectrometer instrument has one sensor looking up (reference, irradiance),
! and the other looks down (radiance).  
! The measurement procedure is:
!   1.  Set the spectrometers measurement range by setting the integration time.
!   2.  Do a dark measurement that subtracts the spectrometer background.
!   3.  Measure the reference looking up and the teflon calibration target looking down (2 files.)
!   4.  Remove the teflon calibration target and do the measurement of the surface in question.
!   5.  Run this program to calculate the surface albedo.
! Modified 13June2014 to add an additional graph output with a broader range.

	PROGRAM albedo
	IMPLICIT none

! Define the variables to hold the raw and normalized spectrometer measurements.
! Measurement definitions:
! index 1 = reference measurement when taking the Teflon spectrum, that is, sensor looking up.
! index 2 = target measurement when taking the Teflon spectrum; that is, looking down.
! index 3 = reference measurement when taking the surface spectrum, that is sensor looking up.
! index 4 = target measurement when taking the surface spectrument, sensor looking down.
	REAL :: RawData(4,10000), wavlen(4,10000)
	REAL :: AveData(4,1000)  ! Spectra normalized by integration time and conformity to 350 nm to 1000 nm steps of 1 nm.

! Filenames used to process the data. filenames() are global to the entire program.
! filenames 1 = reference measurement when taking the Teflon spectrum, that is, sensor looking up.
! filenames 2 = target measurement when taking the Teflon spectrum; that is, looking down.
! filenames 3 = reference measurement when taking the surface spectrum, that is sensor looking up.
! filenames 4 = target measurement when taking the surface spectrument, sensor looking down.
! filenames 5 = output albedo spectrum filename.  Also used to obtain the gnuplot graph.

	CHARACTER*40 :: filenames(10) 

! Get the filenames.
	CALL GetFileNames(filenames)
! Read in the data.
	CALL ReadData(filenames,wavlen,RawData)
! Calculate the spectrally averaged data, AveData() from the RawData().
	CALL SpecAve(wavlen,RawData,AveData)
! Calculate the albedo from the AveData(). Write the albedo to the output file.
	CALL GetAlbedo(filenames,AveData)
! Create a gnuplot gif format graph of the albedo spectrum.  Put it in the output directory.
	CALL MakePlot(filenames)
! Create a gnuplot gif format graph of the albedo spectrum.  Put it in the output directory.
! Second plot has a broader wavelength range.
	CALL MakePlot2(filenames)



	END PROGRAM albedo

! *************************************************************************
! Subroutine to get file names.
! *************************************************************************
! Variables are all defined in the global definitions in the main program.
	SUBROUTINE GetFileNames(filenames)
	CHARACTER*40 :: filenames(10) 

! Open an output directory.
	CALL SYSTEM('mkdir -p output')

	CALL SYSTEM('clear')
	CALL SYSTEM('ls *.txt') ! list the text files that are likely to contain spectra.
	
	WRITE(*,*) 'If an error is made when entering a file name, do a control c command and start over'
	WRITE(*,*) '  '

	WRITE(*,*) 'Enter the reference file name (sensor looking up) for the Teflon measurement.'
	READ(*,'(A)') filenames(1)

	WRITE(*,*) 'Enter the target file name (sensor looking down) for the Teflon measurement.'
	READ(*,'(A)') filenames(2)

	WRITE(*,*) 'Enter the reference file name (sensor looking up) for the surface measurement.'
	READ(*,'(A)') filenames(3)

	WRITE(*,*) 'Enter the target file name (sensor looking down) for the surface measurement.'
	READ(*,'(A)') filenames(4)

	WRITE(*,*) " "

	WRITE(*,*) 'Enter file name for the calibrated surface albedo output.'
	WRITE(*,*) 'Do not enter a suffix like .txt.'
	READ(*,'(A)') filenames(5) ; filenames(5)='output/'//filenames(5)

	WRITE(*,*) " "
	WRITE(*,*) 'Output files can be found in the directory named output'	

	RETURN
	END

! *************************************************************************
! Subroutine to read in the data.
! *************************************************************************
	SUBROUTINE ReadData(filenames,wavlen,RawData)
	IMPLICIT none
	CHARACTER*40 :: filenames(10) 
	REAL :: RawData(4,10000), wavlen(4,10000)
	REAL time,xlam,RawSpec
	INTEGER i,j,k,jj
	CHARACTER*200 :: line  ! local variable to use to read in each line as a character.

! Temporary test file.
	OPEN(11,file='temporary.txt')

	DO j=1,4  

	OPEN(1,FILE=TRIM(filenames(j))) 

! Initialize the data counting variable.
	jj=0

! Loop through the raw data and process it.
	DO i=1,100000

! Read in a line of data.
	READ(1,'(A)',END=100) line  ! Read in a whole line as a char.

	 IF (i.EQ.9) THEN
! Fill the first 25 characters with a blank space.
	DO k=1,25  ;  line(k:k) = " "  ;  END DO ! k=1,25

	READ(UNIT=line,FMT=3) time
3	FORMAT(25x,e6.0)


	time = time / 1.0e6 ! Converts integration time to seconds.

	ELSE IF (line(1:8).EQ.'>>>>>End') THEN
	  GOTO 100
	ELSE IF (i.GT.17) THEN
	  jj=jj+1
	  BACKSPACE(1)
	  READ(1,*) xlam,RawSpec
	  wavlen(j,jj)=xlam
	  RawData(j,jj) =  RawSpec / time
	END IF
	
	END DO ! DO i=1,100000

! Have finished reading in the data from the file.
100	CLOSE(1)

	ENDDO ! DO j=1,4 read of all files.

	RETURN
	END

! *************************************************************************
! Subroutine to average the data to a resolution of 1 nm.
! *************************************************************************
	SUBROUTINE SpecAve(wavlen,RawData,AveData)
	IMPLICIT none
	REAL :: RawData(4,10000), wavlen(4,10000)
	REAL :: AveData(4,1000)  ! Spectra normalized by integration time and conformity to 350 nm to 1000 nm steps of 1 nm.
	INTEGER i,j,jj,k,kount
	
	DO k=1,4  ! Do each measurement.

	DO i=350,1000
	j=i-349
	kount=0
	 AveData(k,j)=0.0
	 DO jj=1,10000
	 IF ((wavlen(k,jj).GE.FLOAT(i)-0.5).AND.(wavlen(k,jj).LE.FLOAT(i)+0.5)) THEN
	   AveData(k,j) = AveData(k,j) + RawData(k,jj)
	   kount=kount+1
	 END IF
	 END DO ! jj=1,10000 loop over raw data

	AveData(k,j) = AveData(k,j) / FLOAT(kount)

	END DO ! i=350,1000

	ENDDO ! k=1,4 loop over each measurement.

	RETURN
	END

! *************************************************************************
! Subroutine to calculate the albedo from the averaged data. Write the data to the output file.
! *************************************************************************

	SUBROUTINE GetAlbedo(filenames,AveData)
	IMPLICIT none
	REAL albedo
	INTEGER i,j,jj,k,kount,kk
	CHARACTER*40 :: filenames(10) 
	REAL :: AveData(4,1000)  ! Spectra normalized by integration time and conformity to 350 nm to 1000 nm steps of 1 nm.

	OPEN(1,FILE=TRIM(filenames(5))//'.txt')
	OPEN(2,FILE='work.trashme')
	WRITE(1,*) 'Wavelength_nm Albedo'  ! Write the header.

	DO i=350,1000
	j=i-349
	albedo = (AveData(4,j)/AveData(3,j)) / (AveData(2,j)/AveData(1,j))
	 DO kk=1,2 ; WRITE(kk,*) FLOAT(i),albedo ; ENDDO
	ENDDO

	CLOSE(1)
	CLOSE(2)

	RETURN
	END


! *************************************************************************
! Subroutine to make a plot of the albedo.
! *************************************************************************
	SUBROUTINE MakePlot(filenames)
	IMPLICIT none

! Input data is from the albedo file created earlier as 'work.trashme'
! This subroutine creates a gnuplot script and then runs it.
	CHARACTER*40 :: filenames(10)

	CHARACTER*40 :: fname  ! local file name, not including the directory.

	INTEGER i
	
	fname=filenames(5)

	DO i=1,7 ; fname(i:i)='' ; ENDDO ! Get rid of the directory in the filename.

	OPEN(1,FILE='PlotScript.gps')
	
	WRITE(1,*) '# write a gnuplot script'
	WRITE(1,*) 'set terminal gif'
	WRITE(1,*) "set output '",TRIM(filenames(5))//".gif'"
	WRITE(1,*) 'set xlabel "Wavelength (nm)"'
	WRITE(1,*) 'set ylabel "Albedo Re Teflon"'
	WRITE(1,*) 'set xrange [400:800]'
	WRITE(1,*) 'set yrange [0:1.2]'
	WRITE(1,*) "plot 'work.trashme' u 1:2 w l title '"//TRIM(ADJUSTL(fname))//"'"
	WRITE(1,*) 'reset'

	CLOSE(1)


! call gnuplot
	CALL SYSTEM('gnuplot PlotScript.gps')

! delete temporary files
!	CALL SYSTEM('rm PlotScript.gps')
!	CALL SYSTEM('rm work.trashme')
	CALL SYSTEM('display '//TRIM(filenames(5))//".gif")

	RETURN
	END	

! *************************************************************************
! Subroutine to make a plot of the albedo.
! *************************************************************************
	SUBROUTINE MakePlot2(filenames)
	IMPLICIT none

! Input data is from the albedo file created earlier as 'work.trashme'
! This subroutine creates a gnuplot script and then runs it.
	CHARACTER*40 :: filenames(10)

	CHARACTER*40 :: fname  ! local file name, not including the directory.

	INTEGER i
	
	fname=filenames(5)

	DO i=1,7 ; fname(i:i)='' ; ENDDO ! Get rid of the directory in the filename.

	OPEN(1,FILE='PlotScript.gps')
	
	WRITE(1,*) '# write a gnuplot script'
	WRITE(1,*) 'set terminal gif'
	WRITE(1,*) "set output '",TRIM(filenames(5))//"_2"//".gif'"
	WRITE(1,*) 'set xlabel "Wavelength (nm)"'
	WRITE(1,*) 'set ylabel "Albedo Re Teflon"'
	WRITE(1,*) 'set xrange [350:950]'
	WRITE(1,*) 'set yrange [0:1.2]'
	WRITE(1,*) "plot 'work.trashme' u 1:2 w l title '"//TRIM(ADJUSTL(fname))//"'"
	WRITE(1,*) 'reset'

	CLOSE(1)


! call gnuplot
	CALL SYSTEM('gnuplot PlotScript.gps')

! delete temporary files
!	CALL SYSTEM('rm PlotScript.gps')
!	CALL SYSTEM('rm work.trashme')
	CALL SYSTEM('display '//TRIM(filenames(5))//"_2"//".gif")

	RETURN
	END	





