#CoPilot AI Assistant.
# Write a python program that reads in data from files in a folder having file that start with the same prefix name followed by time. The prefix name needs to be an input to the script from the program user. Each file has 20 lines of header to be ignored, and then two columns of data containing wavelength in one column and Transmission in the other column. Calculate OpticalDepth using the equation OpticalDepth=-ln(Transmission/100). The first optical depth calculation is the pure water optical depth. Subtract this optical depth from each file. Make individual plots of the OpticalDepth data, and a plot that overlays all of the files in one plot. Label the x axis with "Wavelength (nm)" and the y axis as "Optical Depth". Save each plot as a .png file. Make an mp4 movie from the individual files and save it. For the individual OpticalDepth graphs, use the same filename as the input files though with the png output suffix. Name the overlay png graph with the prefix name followed by the suffix .png. Make all plots 10" x 10" and 300 dpi. Name the mp4 movie with the prefix name followed by the .mp4 suffix. Output all of the data in a single file in excel format using the prefix as a name, the first data column as the wavelength in one file and all of the other columns OpticalDepth. Make the file header have wavelength as the first column label followed by the text between the prefix and the '.' in the filename as the label for each column.
# Debugged by PA.
# Install ffmpeg from https://ffmpeg.org/download.html
# install numpy pandas matplotlib

# Notes:
# Change line 101, plt.title, for other data.
# This script assumes the first file is for 'pure water' and the other files
# may contain particles or other things in the water. 
# If subtraction is not wanted, comment out lines 68-72.
# The pure water spectrum is subtracted from the others.

import glob
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib.animation as animation
from scipy.signal import savgol_filter

def smooth_spectrum(wavelength, spectrum, window_length=11, polyorder=2):
    """
    Apply Savitzky-Golay smoothing to the input spectrum.

    Parameters:
    - wavelength: array-like, Wavelength values
    - spectrum: array-like, Spectrum values
    - window_length: int, optional, The length of the filter window (must be odd)
    - polyorder: int, optional, The order of the polynomial used to fit the samples

    Returns:
    - smoothed_spectrum: array-like, Smoothed spectrum values
    """

    # Ensure window_length is odd and less than the size of the input array
    if window_length % 2 == 0 or window_length >= len(spectrum):
        raise ValueError("window_length must be odd and less than the length of the spectrum.")
        
    smoothed_spectrum = savgol_filter(spectrum, window_length, polyorder)
    
    return smoothed_spectrum



def process_files(prefix):
    files = sorted(glob.glob(f'{prefix}*.txt'))
    if not files:
        print(f"No files found with prefix {prefix}")
        return

    all_wavelengths = []
    all_optical_depths = []
    header_labels = ["Wavelength_nm"]
    
#    print(files)

    pure_water_optical_depth = []
    for file in files:
        data = np.loadtxt(file, skiprows=584)
        wavelengths =  data[:, 0]
        transmission = data[:, 1]
        optical_depth_raw = -np.log(transmission / 100)
        optical_depth = smooth_spectrum(wavelengths,optical_depth_raw)



# Use the first file's optical depth as the pure water optical depth
        if len(all_optical_depths) == 0:
            pure_water_optical_depth = optical_depth
 #           print (pure_water_optical_depth)
#        optical_depth -= pure_water_optical_depth
        optical_depth = optical_depth - pure_water_optical_depth
#        print(pure_water_optical_depth)
         

        all_wavelengths.append(wavelengths)
        all_optical_depths.append(optical_depth)
        header_labels.append(file.split('.')[-2].replace(prefix, ""))

        # Generate individual plot
        fig, ax = plt.subplots(figsize=(10, 10))
        ax.plot(wavelengths, optical_depth)
        ax.set_xlabel("Wavelength (nm)", fontsize=14, fontweight='bold')
        ax.set_ylabel("Optical Depth", fontsize=14, fontweight='bold')
        plt.xticks(fontsize=18, fontweight='bold')
        plt.yticks(fontsize=18, fontweight='bold')
        plt.title(file, fontsize=24, fontweight='bold')
        plt.savefig(file.replace(".txt", ".png"), dpi=300)
        plt.close()

    # Generate overlay plot
    fig, ax = plt.subplots(figsize=(10, 10))
    for wavelengths, optical_depth in zip(all_wavelengths, all_optical_depths):
        ax.plot(wavelengths, optical_depth)

    ax.set_xlabel("Wavelength (nm)", fontsize=18, fontweight='bold')
    ax.set_ylabel("Optical Depth", fontsize=18, fontweight='bold')
    plt.xticks(fontsize=18, fontweight='bold')
    plt.yticks(fontsize=18, fontweight='bold')
#    plt.title(f"Overlay of {prefix} files", fontsize=24, fontweight='bold')
    plt.title("Whole milk: 0-18 drops: 2/10/2025", fontsize=24, fontweight='bold')
    plt.savefig(f"{prefix}.png", dpi=300)
    plt.show()
    plt.close()

    # Create mp4 movie
    fig, ax = plt.subplots(figsize=(10, 10))
    line, = ax.plot([], [], lw=2)
    ax.set_xlim(min(min(w) for w in all_wavelengths), max(max(w) for w in all_wavelengths))
    ax.set_ylim(min(min(od) for od in all_optical_depths), max(max(od) for od in all_optical_depths))
    ax.set_xlabel("Wavelength (nm)", fontsize=18, fontweight='bold')
    ax.set_ylabel("Optical Depth", fontsize=18, fontweight='bold')

    def init():
        line.set_data([], [])
        return line,

    def animate(i):
        line.set_data(all_wavelengths[i], all_optical_depths[i])
        return line,

    ani = animation.FuncAnimation(fig, animate, init_func=init, frames=len(all_wavelengths), interval=200, blit=True, repeat=True, repeat_delay=500)
    ani.save(f"{prefix}.mp4", writer='ffmpeg', fps=2)

    # Output data to Excel
    data_dict = {"wavelength": all_wavelengths[0]}
    for i, optical_depth in enumerate(all_optical_depths):
        label = header_labels[i + 1]
        data_dict[label] = optical_depth

    df = pd.DataFrame(data_dict)
    df.to_excel(f"{prefix}.xlsx", index=False)

if __name__ == "__main__":
   # prefix = input("Enter the file prefix: ")d
    prefix = "Trans_Transmission_"
    process_files(prefix)
