/* 
 Program to data log onto SD card the following:
 Sensirion SHT75 temperature and relative humidity measurements.
 CO2 measurement as an analog signal from the surplus sensor.
 Date and time from the Chronodot device.
 Saves data to the SD card about every 5 seconds.
*/
#include <SD.h>
#include <Sensirion.h>
#include <Wire.h>

// Pins used for the Sensirion SHT75 sensor.
const uint8_t dataPin  =  6; // These are the digital pins 6 and 7 on the Teensy.
const uint8_t clockPin =  7;

// These are global variables, available to all functions and procedures.
int seconds;  //  00-59 
int minutes;  //  00-59 
int hours;    //  1-12 - 00-23 
int days;     //  1-7 assuming Sunday is day 1.
int date;     //  01-31
int months;   //  01-12
int years;    //  0-99 

char fname_data[] = "Sounding.txt" ;

// header is the list of things to be written at the top of all files.
// This is written every time the instrument starts so it is possible to track that.
  String topOfFile = "Date_PDT\tTime_PDT\tT_C\tTdew_C\tRH_%\tP_mb\te_mb\trhoWV_gPerCuMet\tCO2_ppmRAW\tCO2_ppmCal\tCO_kOhms\tCO_ppm\tModComEff\tVcc_mVolts\tb_calfac\ta_calfac" ;
  String serialnum = "ARL3 is the serial number for this card. Each sensor receives individual calibration." ; 

// Variables to be read from Sensirion sht75.
float temperature;
float humidity;
float dewpoint;

Sensirion tempSensor = Sensirion(dataPin, clockPin);

 File data;
 char filename[] = "headT.txt"; // We will write the column headings for the data in the SD card.

void setup() // every command here executes once when the Arduino turns on - good for creating files and setting up pins
{
   pinMode(10,OUTPUT); // necessary for SD shield.  This is the CS pin on the SPI port. Pins 10, 11, 12, and 13 are used on Teensy.
  if(!SD.begin(10)) // initialize SD shield
  {
    // normally here we would throw an error warning if the SD card couldn't be initialized. You can add code to light up a red LED, but otherwise you can leave this empty.
  }
  data = SD.open(filename, FILE_WRITE); // open the data file, or, if it doesn't exist, create it.
  data.println(topOfFile); // the file header
  data.println(serialnum); // serial number for this device.
  data.close();
  
  // Set up the serial port monitor to write output for computer or other monitors.
  Serial.begin(9600);
  
  // Set up the I2C connection for the chronodot2.1 clock.
  Wire.begin();
  
  // Set up the analog read characteristics.
  analogReadAveraging(32);  // Do 32 measurements and average them for every analog input measurement.
  analogReadResolution(16); // Do 16 bit analog read resolution, if available this would be great.
  
}   // End void setup().

void loop()   // This is a never ending loop. //
{
   
  // Define the CO, CO2 and pressure measurements. //
  float    c02;
  float    c02c;  // Calibrated CO2 = c02 * 1013.25 * (Temperature+273.15) / (Pressure * 273.15) 
  float    CO;
  float    RCO; // Resistance measured for CO sensor.
  float    sig; // Conductivity of the CO sensor, sig=1/RCO.
  float    Vcc;
  float    norm; 
  float   rhoWV ; // Water vapor density in g/m3.
  float   esTdew ; // Water vapor pressure in mb unit.
  float   b ; // temporary variable to get the water vapor part of the CO sensor calibration.
  float   a ; // calibration value for the CO sensor based on gas with known concentration calibration.
  float rhoCO ; // CO density in mg / m3 used to get the CO concentration.
  float modComEff ; // Modified combustion efficiency = deltaCO2 / (deltaCO2 + CO) and all in common ppm units.
  double   Press ; // pressure in mb computed from Press.

  // Analog data acquisition section.
  // Average signals over about 5 seconds to allow for proper operation of the T and RH sensor digital sensor.
  //
  // Do the c02, temperature, and RH measurements. // 
  tempSensor.measure(&temperature, &humidity, &dewpoint);
  // telaire 6004a module: 2000 ppm = 4 volts. scale factor = 2000 ppm * analogRead counts  * 3.3 volts / (4.0 volts * 65535 counts) ,
  // so the equation is CO2 in ppm = analogReadCounts / 39.72  .  The upper range will be 1650 ppm with the 3.3 volt a/d system.
  // Note:  C02 reports in ppm assuming a certain pressure and temperature.  It has to be converted to local conditions with T and P measurements.
  // We start with the normalization of the voltage.  Vcc should be 2.5 volts as a voltage divider is used.
  // Then norm = 2.5 Volts / Vcc corrects the analog voltages to expected values for the sensors.
  // CO sensor output is reported in kOhms.
  Vcc   = 0.0 ; 
  Press = 0.0 ;
  RCO   = 0.0 ;
  c02   = 0.0 ;
  for (int i=0; i <= 999; i++){ 
  delay(2) ; 
  Vcc     += (float)(analogRead(3)) * 0.050355    ; // Card voltage in millivolts.
  //  delay(1) ; 
  // Do the pressure measurement in mb.
  Press += 105.6 + analogRead(2) / 57.83  ;  // Calculate pressure based on calibration factor for MPX4115 transducer.
  // CO Voltage out = 5000 mVolts * 10 kOhm / (10 kOhm + Rsensor) where Rsensor is the electrochemical
  // sensor resistance.  Voltage is roughly proportional to CO, with an offset. 
  delay(1) ; 
  RCO     += (float)(analogRead(1)) * 0.050355    ;  // Vout for CO sensor in millivolts.
  delay(1) ;    
  c02    += (float)(analogRead(0)) / 39.72    ; 
  }
  Vcc=Vcc / 1000.0 ;               // Vcc appears here in millivolts.
  norm = 2500.0 / Vcc  ; 
  Press=Press * norm / 1000.00 ;   // Pressure units are mb.
  RCO = 10.0 * (2000.0 * Vcc - RCO) / RCO ; // RCO = RL * (Vcc - Vout)/Vout, RL=10kOhms.
  sig = 10000.0 / RCO ;  // Conductivity of the CO SnO2 sensor. in unusual units for simplicity.
  c02 = c02 / 1000.00 ; 
  c02c = c02 * 3.7095 * (temperature + 273.15) / Press ; // Calibrated CO2 based on air density correction.
  esTdew = 6.11 * pow(10,7.5*dewpoint/(237.3 + dewpoint)) ;
  rhoWV = esTdew * 216.5 / (temperature + 273.15) ;
  // b = sig / rhoWV ;  // Calibration constant for the water vapor response of the CO sensor. Needed for each instrument.
  b = 19.4 ; // for the ARL 3 serial number sensor .
  a = 19.4 ;  // Calibration constant used to get correct CO concentration.  Needs to be determined and set for each instrument.
  CO = 2.97 * (temperature + 273.15) * (sig - b * rhoWV) / (a * Press) ;  // CO in calibrated ppm units for the given pressure and temperature.
  modComEff = (c02c - 400.0) / (c02c - 400.0 + CO)  ; // Background CO2 assumed to be 400 ppm.
  
  
  // Get the actual time now, from Chronodot 2.1. // Variables are filled due to the global definitions of variable types.
  get_time();
  get_date();

 // Write the data to file. ;
  data = SD.open(fname_data, FILE_WRITE);
  digitalClockDisplaySD(); // Writes the time and date.
  data.print("\t");
  data.print(temperature);
  data.print("\t");
  data.print(dewpoint);
  data.print("\t");
  data.print(humidity);
  data.print("\t");
  data.print(Press);
  data.print("\t");
  data.print(esTdew);
  data.print("\t");
  data.print(rhoWV);
  data.print("\t"); 
  data.print(c02);
  data.print("\t");
  data.print(c02c);
  data.print("\t");
  data.print(RCO);
  data.print("\t");
  data.print(CO);
  data.print("\t");  
  data.print(modComEff);
  data.print("\t");
  data.print(Vcc);
  data.print("\t");
  data.print(b);
  data.print("\t");
  data.println(a);
  data.close();

 // Write the data to the serial port too, to look at it. ;
  digitalClockDisplay(); // Writes the time and date.
  Serial.print("\t");
  Serial.print("ElapsedSeconds=" );                       
  Serial.print(millis()/1000); // basic timestamp: returns number of seconds since microcontroller was turned on - MAKE A NOTE OF THIS TIME.
  Serial.print("\tT_C=") ;
  Serial.print(temperature)  ;      
  Serial.print("\tTdew_C=") ; 
  Serial.print(dewpoint)   ;      
  Serial.print("\tRH_%=") ;
  Serial.print(humidity)  ;      
  Serial.print("\tPressure_mb=") ;
  Serial.print(Press) ;
  Serial.print("\teSatTdew_mb=") ;
  Serial.print(esTdew) ;
  Serial.print("\trhoWV_gPerCuMet=") ;
  Serial.print(rhoWV) ; 
  Serial.print("\tCO2_ppmRaw=") ;
  Serial.print(c02)  ;
  Serial.print("\tCO2_ppmCal=") ;
  Serial.print(c02c)  ;
  Serial.print("\tCO_kOhms=") ;
  Serial.print(RCO)  ; 
  Serial.print("\tCO_ppm=") ;
  Serial.print(CO)  ; 
  Serial.print("\tModifiedComEff") ;
  Serial.print(modComEff) ;
  Serial.print("\tVcc_mVolts=") ;
  Serial.print(Vcc)  ; 
  Serial.print("\tb_calFac=") ;
  Serial.print(b)  ; 
  Serial.print("\ta_calFac=") ;
  Serial.println(a)  ;  
  
//  delay(5000); // the SHT7x needs to wait at least 3.5 seconds between readings or it will overheat.

}  // end void loop

// SD card printing of actual time and date.
// Write the digital clock output to the SD card writer.
void digitalClockDisplaySD(){
  SDPrintDigits(months) ;
  data.print("/");
  SDPrintDigits(date) ; 
  data.print("/20") ;
  SDPrintDigits(years) ;
  data.print(" ") ;
  SDPrintDigits(hours) ;
  data.print(":") ;
  SDPrintDigits(minutes) ;
  data.print(":") ;
  SDPrintDigits(seconds) ;
  data.print("\t") ;
}
  
void SDPrintDigits(int digits){
  if(digits < 10)
    data.print("0") ; 
  data.print(digits) ;
}

void get_date()
{
  Wire.beginTransmission(104); 
  Wire.write(3);//set register to 3 (day)
  Wire.endTransmission();
  Wire.requestFrom(104, 4); //get 5 bytes(day,date,month,year,control);
  days   = bcdToDec(Wire.read());
  date  = bcdToDec(Wire.read());
  months = bcdToDec(Wire.read());
  years  = bcdToDec(Wire.read());
}

void get_time()
{
  Wire.beginTransmission(104); 
  Wire.write(0);//set register to 0
  Wire.endTransmission();
  Wire.requestFrom(104, 3);//get 3 bytes (seconds,minutes,hours);
  seconds = bcdToDec(Wire.read() & 0x7f);
  minutes = bcdToDec(Wire.read());
  hours = bcdToDec(Wire.read() & 0x3f);
}
    
///////////////////////////////////////////////////////////////////////

byte decToBcd(byte val)
{
  return ( (val/10*16) + (val%10) );
}

byte bcdToDec(byte val)
{
  return ( (val/16*10) + (val%16) );
}
//////////////////////////////////////////////////////////////////////

// Write the digital clock output to the serial port.  Can modify for the SD card writer.
void digitalClockDisplay()
{
  SerialPrintDigits(months) ;
  Serial.print("/");
  SerialPrintDigits(date) ; 
  Serial.print("/20") ;
  SerialPrintDigits(years) ;
  Serial.print(" ") ;
  SerialPrintDigits(hours) ;
  Serial.print(":") ;
  SerialPrintDigits(minutes) ;
  Serial.print(":") ;
  SerialPrintDigits(seconds) ;
}
  
void SerialPrintDigits(int digits)
{
  if(digits < 10)
    Serial.print("0") ; 
  Serial.print(digits) ;
}



    
